<?php

/*
  Title:        SrPago Payment Gateway
  Author:       Roberto Ramírez
  URL:          https://www.srpago.com
  License:
  License URI:
*/

/**
 * Clase principal para la integración del plugin.
 */
class WC_Gateway_SrPagoCard extends WC_SrPagoCard
{
    /**
     * Monto mínimo para habilitar los pagos diferidos.
     */
    const MONTHLY_INSTALLMENTS_MIN_AMOUNT = 500;

    protected $GATEWAY_NAME = 'SrPago Cards';

    /**
     * Inicializar la configuración del plugin.
     */
    public function __construct()
    {
        $this->id                   = 'WC_Gateway_SrPagoCard';
        $this->has_fields           = true;
        $this->method_title         = __('SrPago Tarjetas', 'SrPagoPaymentGateway');
        $this->method_description   = __('SrPago - Integración de pagos con tarjetas de crédito y debito', 'SrPagoPaymentGateway');

        $this->checkRequirements();
        $this->init_form_fields();
        $this->init_settings();

        $this->enabled      = $this->get_option('enabled');
        $this->title        = $this->get_option('title');

        $this->sandbox      = strcmp($this->get_option('sandbox'), 'yes') == 0;
        $this->appKey       = $this->sandbox ? $this->get_option('sandbox_app_key') : $this->get_option('live_app_key');
        $this->appSecret    = $this->sandbox ? $this->get_option('sandbox_app_secret') : $this->get_option('live_app_secret');
        $this->publicKey    = $this->sandbox ? $this->get_option('sandbox_public_key') : $this->get_option('live_public_key');

        $this->logging      = 'yes' === $this->get_option('logging');

        parent::__construct();

        add_action('wp_enqueue_scripts', array($this, 'enqueue_payment_scripts'));
    }

    /**
     * Inicializar los campos de configuración.
     *
     * Añadir un array de campos que se mostrarán
     * en la pantalla de configuración del plugin.
     */
    public function init_form_fields()
    {
        $this->form_fields = include('includes/cards_settings.php');

        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
    }

    /**
     * Muestra los campos de configuración.
     *
     * Se sobreescribe el método para mostrar los mensajes de error cuando
     * los requerimientos no se cumplen al 100%.
     */
    public function admin_options()
    {
        include_once('includes/admin_settings.php');
    }

    /**
     * Incluir los campos para el pago con tarjeta.
     */
    public function payment_fields()
    {
        global $woocommerce;

        $monthly_installments_available = (
            (0 == strcmp($this->get_option('monthly_installments_available'), 'yes')) &&
            (floatval($this->get_option('monthly_installments_min_amount')) <= floatval($woocommerce->cart->total))
        );
        $monthly_installments_available_months = $this->get_option('monthly_installments_available_months');

        $this->form_fields = include('includes/cards_checkout_form.php');
    }

    /**
     * Procesa el pago
     *
     * La variable $_POST contiene la información del formulario de pago,
     * si el pago fue aprobado regreso un array con la siguiente estructura.
     *
     *        return array(
     *            'result'   => 'success',
     *            'redirect' => $this->get_return_url( $order )
     *        );
     *
     * @param int $order_id Id de la orden de WooCommerce
     * @return array
     */
    public function process_payment($order_id)
    {
        global $woocommerce;

        $this->order = new WC_Order($order_id);

        if (! isset($_POST['srpago_token'])) {
            $this->wcAddNotice(__('Ha ocurrido un error en los parametros que se enviarón, favor de volver a intentar.', 'SrPagoPaymentGateway'));

            return;
        }

        $this->initSrPago();

        $srpago_token = $_POST['srpago_token'];
        $monthly_installments = intval($_POST['monthly_installments']);

        if ($this->processCardCharge($srpago_token, $monthly_installments)) {
            $woocommerce->cart->empty_cart();

            $this->order->payment_complete();
            $this->order->add_order_note(sprintf(
                "%s payment completed with Transaction Id of '%s'",
                $this->GATEWAY_NAME,
                $this->transaction_id
            ));

            return array(
                'result' => 'success',
                'redirect' => $this->get_return_url($this->order)
            );
        } else {
            $this->order->update_status('failed');

            $this->order->add_order_note(sprintf(
                "%s Credit Card Payment Failed : '%s'",
                $this->GATEWAY_NAME,
                $this->transaction_error_message
            ));

            return array();
        }
    }

    public function validate_monthly_installments_min_amount_field($key, $value)
    {
        if (self::MONTHLY_INSTALLMENTS_MIN_AMOUNT > floatval($value)) {
            WC_Admin_Settings::add_error(esc_html__('El monto mínimo para pagos diferidos es de $' . self::MONTHLY_INSTALLMENTS_MIN_AMOUNT, 'SrPagoPaymentGateway'));

            throw new Exception('Invalid value: {$value}');
        }

        return $value;
    }
}

add_filter('woocommerce_payment_gateways', array('WC_Gateway_SrPagoCard', 'addGateway'));
