<?php

/*
  Title:        SrPago Payment Gateway
  Author:       Roberto Ramírez
  URL:          https://www.srpago.com
  License:
  License URI:
*/

/**
 * Clase principal para la integración del plugin.
 */
class WC_Gateway_SrPagoStore extends WC_SrPagoStore
{
    protected $GATEWAY_NAME = 'SrPago Stores';

    /**
     * Inicializar la configuración del plugin.
     */
    public function __construct()
    {
        $this->id                   = 'WC_Gateway_SrPagoStore';
        $this->has_fields           = true;
        $this->method_title         = __('SrPago en efectivo', 'SrPagoPaymentGateway');
        $this->method_description   = __('SrPago - Integración de pagos en efectivo', 'SrPagoPaymentGateway');

        $this->checkRequirements();
        $this->init_form_fields();
        $this->init_settings();

        $this->enabled              = $this->get_option('enabled');
        $this->title                = $this->get_option('title');
        $this->stores_available     = $this->get_option('stores_available');

        $this->sandbox              = strcmp($this->get_option('sandbox'), 'yes') == 0;
        $this->appKey               = $this->sandbox ? $this->get_option('sandbox_app_key') : $this->get_option('live_app_key');
        $this->appSecret            = $this->sandbox ? $this->get_option('sandbox_app_secret') : $this->get_option('live_app_secret');
        $this->publicKey            = $this->sandbox ? $this->get_option('sandbox_public_key') : $this->get_option('live_public_key');

        $this->logging              = 'yes' === $this->get_option('logging');

        parent::__construct();

        add_action('wp_enqueue_scripts', array($this, 'enqueue_payment_scripts'));
        add_action('woocommerce_api_srpago_webhook_listener', array($this, 'webhook_handler'));
    }

    /**
     * Inicializar los campos de configuración.
     *
     * Añadir un array de campos que se mostrarán
     * en la pantalla de configuración del plugin.
     */
    public function init_form_fields()
    {
        $this->form_fields = include('includes/stores_settings.php');

        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
    }

    /**
     * Muestra los campos de configuración.
     *
     * Se sobreescribe el método para mostrar los mensajes de error cuando
     * los requerimientos no se cumplen al 100%.
     */
    public function admin_options()
    {
        include_once('includes/admin_settings.php');
    }

    /**
     * Incluir los campos para el pago en efectivo.
     */
    public function payment_fields()
    {
        $this->form_fields = include('includes/stores_checkout_form.php');
    }

    /**
     * Procesa el pago
     *
     * La variable $_POST contiene la información del formulario de pago,
     * si el pago fue aprobado regreso un array con la siguiente estructura.
     *
     *        return array(
     *            'result'   => 'success',
     *            'redirect' => $this->get_return_url( $order )
     *        );
     *
     * @param int $order_id Id de la orden de WooCommerce
     * @return array
     */
    public function process_payment($order_id)
    {
        global $woocommerce;

        $this->order = new WC_Order($order_id);

        $store = array_key_exists('store', $_POST) ? $_POST['store'] : null;

        if (! isset($this->supported_stores[$store])) {
            $this->wcAddNotice(__('Ha ocurrido un error en los parametros que se enviarón, favor de volver a intentar.', 'SrPagoPaymentGateway'));

            return;
        }

        $this->initSrPago();

        if ($this->request_payment_reference($store)) {
            $this->order->update_status('on-hold', 'En espera de pago');

            wc_reduce_stock_levels($this->order->get_id());

            $woocommerce->cart->empty_cart();

            return array(
                'result' => 'success',
                'redirect' => $this->get_return_url($this->order),
            );
        } else {
            $this->order->add_order_note(sprintf(
                "%s Credit Card Payment Failed : '%s'",
                $this->GATEWAY_NAME,
                $this->transaction_error_message
            ));

            return array();
        }
    }
}

add_filter('woocommerce_payment_gateways', array('WC_Gateway_SrPagoStore', 'addGateway'));
