<?php

/*
  Title:        SrPago Payment Gateway
  Author:       Roberto Ramírez
  URL:          https://www.srpago.com
  License:
  License URI:
*/

if (! class_exists('SrPago')) {
    require_once __DIR__ . '/library/srpago_php/init.php';
}

use SrPago\SrPago;

class WC_SrPago extends WC_Payment_Gateway
{
    /**
     * Version del plugin
     *
     * @var string
     */
    const WC_SRPAGO_VERSION = '1.2.0';

    /**
     * Version minima de PHP
     *
     * @var string
     */
    const WC_SRPAGO_MIN_PHP_VER = '5.3.3';

    /**
     * Contenedor de los requerimientos que no se cumplen.
     *
     * @var array
     */
    protected $requirement_errors = [];

    /**
     * Instancia de la orden
     *
     * @var object
     */
    protected $order;

    /**
     * Id de la transacción de SrPago.
     *
     * @var string
     */
    protected $transaction_id = null;

    /**
     * Mensaje de error de la transacción.
     *
     * @var string
     */
    protected $transaction_error_message = null;

    /**
     * Indica si se registran los mensajes de depuración.
     *
     * @var bool
     */
    protected $logging;

    /**
     * @var object
     */
    private $logger;

    /**
     * Agrega la hoja de estilo para el formulario de pago.
     */
    public function __construct()
    {
        wp_enqueue_style('srpago', plugins_url('assets/css/srpago.css', __FILE__), '', '0.0.3', 'screen');
    }

    /**
     * Inicializa los parametros por default del SDK.
     */
    protected function initSrPago()
    {
        SrPago::setLiveMode(! $this->sandbox);
        SrPago::setApiKey($this->appKey);
        SrPago::setApiSecret($this->appSecret);
    }

    /**
     * Registra el gateway en WooCommerce.
     *
     * @return array()
     */
    public static function addGateway($methods)
    {
        $methods[] = get_called_class();

        return $methods;
    }

    /**
     * Agrega un mensaje para ser mostrado al cliente.
     *
     * @param string $message
     */
    protected function wcAddNotice($message)
    {
        global $woocommerce;
        global $wp_version;

        if (version_compare($wp_version, '4.1', '>=')) {
            wc_add_notice(__('Error: ', 'woothemes') . $message , $notice_type = 'error');
        } else {
            $woocommerce->add_error(__('Error: ', 'woothemes') . $message);
        }
    }

    /**
     * Valida los requerimientos del plugin, en caso de que no se cumplan al
     * 100% los campos de configuración apareceran desabilitados.
     */
    protected function checkRequirements()
    {
        // Ejemplo:
        // if (! function_exists('mcrypt_encrypt')) {
        //     $this->requirement_errors[] = __('<strong>ERROR:</strong> La extensión <strong><u>mcrypt</u></strong> no se encuentra instalada.', 'SrPagoPaymentGateway');
        // }
    }

    /**
     * Maneja las notificaciones webhooks
     *
     * @return null
     */
    public function webhook_handler()
    {
        header('HTTP/1.1 200 OK');

        $body = file_get_contents('php://input');
        $data = json_decode($body, true);

        if (isset($data['operation']) && 'POS' != $data['operation']['payment_method']) {
            try {
                $data = $data['operation'];

                $this->log('debug', wc_print_r([ 'srpago.webhook' => $data, ], true));

                $order_id = $data['reference']['number'];

                $order = new WC_Order($order_id);

                if ($order) {
                    update_post_meta($order->get_id(), 'srpago-payment-date', $data['timestamp']);

                    $order->payment_complete();

                    $order->add_order_note(sprintf(
                        "%s payment completed with Transaction Id of '%s'",
                        $this->GATEWAY_NAME,
                        $data['transaction']
                    ));
                }
            } catch (\Exception $e) {
                $this->log(
                    'critical',
                    wc_print_r([
                        'message' => 'Error processing webhook notification',
                        'exception' => $e,
                    ], true)
                );
            }
        }

        return wp_send_json_success([]);
    }

    /**
     * Maneja los mensajes de error que se muestran al cliente.
     *
     * @param \SrPago\Error\SrPagoError $e
     * @return string
     */
    protected function buildError(\SrPago\Error\SrPagoError $e)
    {
        $error = $e->getError();
        $message = null;

        switch ($error['code']) {
            case 'InvalidParamException':
            case 'InvalidEncryptionException':
            case 'PaymentFilterException':
            case 'SwitchException':
            case 'InternalErrorException':
            default:
                $message = 'Ha ocurrido un error al procesar la transacción, favor de volverlo a intentar mas tarde.';
            break;
            case 'SuspectedFraudException':
            case 'InvalidTransactionException':
            case 'PaymentException':
                $message = 'La transacción ha sido denegada por el banco emisor.';
            break;
        }

        return $message;
    }

    /**
     * Guarda un mensaje de log.
     *
     * @param string $level
     * @param string $message
     * @return
     */
    public function log($level, $message)
    {
        if (! $this->logging) {
            return;
        }

        $level = mb_strtolower($level);
        $levels = array( 'emergency', 'alert', 'critical', 'error', 'warning', 'notice', 'informational', 'debug', );

        if (! in_array($level, $levels)) {
            return;
        }

        if (empty($this->logger)) {
            $this->logger = wc_get_logger();
        }

        $this->logger->log($level, $message, [ 'source' => 'woocommerce-gateway-srpago', ]);
    }
}
