<?php

/*
  Title:        SrPago Payment Gateway
  Author:       Roberto Ramírez
  URL:          https://www.srpago.com
  License:
  License URI:
*/

use SrPago\Error\SrPagoError;
use SrPago\Http\HttpClient;
use SrPago\SrPago;
use SrPago\Token;

class WC_SrPagoCard extends WC_SrPago
{
    /**
     * Contruye la metadata de la transacción.
     *
     *  @return array()
     */
    protected function buildMetadata()
    {
        $metadata = array();

        $metadata['orderMessage'] = $this->order->get_customer_note();

        $metadata['billing'] = array(
            'billingEmailAddress'   => $this->order->get_billing_email(),
            'billingFirstName-D'    => $this->order->get_billing_first_name(),
            'billingLastName-D'     => $this->order->get_billing_last_name(),
            'billingAddress-D'      => $this->order->get_billing_address_1(),
            'billingAddress2-D'     => $this->order->get_billing_address_2(),
            'billingCity-D'         => $this->order->get_billing_city(),
            'billingState-D'        => $this->order->get_billing_state(),
            'billingPostalCode-D'   => $this->order->get_billing_postcode(),
            'billingCountry-D'      => $this->order->get_billing_country(),
            'billingPhoneNumber-D'  => $this->order->get_billing_phone(),
        );

        $order_coupons = $this->order->get_items('coupon');
        foreach($order_coupons as $index => $coupon) {
            $metadata['coupons']['coupon'][] = array(
                'couponCode'    => $coupon['name'],
                'couponAmount'  => $coupon['discount_amount'],
            );
        }

        $user = $this->order->get_user();
        $metadata['member'] = array(
            'memberId'              => $this->order->get_user_id(),
            'membershipDate'        => $user->user_registered,
            'memberFullName'        => $user->display_name,
            'memberEmailAddress'    => $user->user_email,
            'membershipStatus'      => ! $user->user_status,
        );

        // Shipping Lines
        $shipping_method = $this->order->get_shipping_method();
        if (! empty($shipping_method)) {
            $metadata['shipping'] = array(
                'shippingCharges'       => (float) $this->order->get_total_shipping(),
                'shippingFirstName'     => esc_html($this->order->get_shipping_first_name()),
                'shippingLastName'      => esc_html($this->order->get_shipping_last_name()),
                'shippingAddress'       => esc_html($this->order->get_shipping_address_1()),
                'shippingAddress2'      => esc_html($this->order->get_shipping_address_2()),
                'shippingCity'          => esc_html($this->order->get_shipping_city()),
                'shippingState'         => esc_html($this->order->get_shipping_state()),
                'shippingPostalCode'    => esc_html($this->order->get_shipping_postcode()),
                'shippingCountry'       => esc_html($this->order->get_shipping_country()),
                'shippingMethod'        => $shipping_method,
            );
        }

        $items = $this->order->get_items();
        foreach ($items as $item) {
            $subTotal    = floatval($item['line_subtotal']);
            $subTotal    = $subTotal / floatval($item['qty']);
            $productmeta = new WC_Product($item['product_id']);
            $sku         = $productmeta->get_sku();
            $unit_price  = $subTotal;
            $itemName    = $item['name'];
            $unitPrice   = intval(round(floatval($unit_price) / 10), 2);
            $quantity    = intval($item['qty']);

            $metadata['items']['item'][] = array(
                'itemNumber' => $sku,
                'itemDescription' => $itemName,
                'itemPrice' => $unit_price,
                'itemQuantity' => $quantity,
            );
        }

        return $metadata;
    }

    /**
     * Procesa el pago
     *
     * Integración con SrPago para procesar el pago.
     *
     *  @return boolean
     */
    protected function processCardCharge($srpago_token, $monthly_installments)
    {
        $log_data = array();

        try {
            $chargeRequest = array(
                'amount' => $this->order->get_total(),
                'description' => sprintf('Charge for %s', $this->order->get_billing_email()),
                'reference' => $this->order->get_id(),
            );

            $monthly_installments_available = (
                (0 == strcmp($this->get_option('monthly_installments_available'), 'yes')) &&
                in_array($monthly_installments, $this->get_option('monthly_installments_available_months')) &&
                (floatval($this->get_option('monthly_installments_min_amount')) <= floatval($this->order->get_total()))
            );

            if ($monthly_installments_available) {
                $chargeRequest['months'] = $monthly_installments;
            }

            $chargeRequest['metadata'] = $this->buildMetadata();
            $chargeRequest['source'] = $srpago_token;

            $log_data['request'] = $chargeRequest;

            $SrPagoCharge = SrPago::Charges();
            $chargeResponse = $SrPagoCharge->create($chargeRequest);

            $chargeResponse = SrPagoMisc::parseResponse($chargeResponse, HttpClient::$last);

            $log_data['response'] = $chargeResponse;
            $this->log('debug', wc_print_r($log_data, true));

            $this->transaction_id = $chargeResponse['transaction'];

            update_post_meta($this->order->get_id(), 'srpago-transaction-id', $this->transaction_id);
            update_post_meta($this->order->get_id(), 'srpago-autorization-code', $chargeResponse['authorization_code']);

            if ($monthly_installments_available) {
                update_post_meta($this->order->get_id(), 'meses-sin-intereses', $monthly_installments);
            }

            return true;
        } catch (SrPagoError $e) {
            $log_data['response'] = $e;
            $this->log('notice', wc_print_r($log_data, true));

            $error = $this->buildError($e);

            $this->wcAddNotice($error);

            $this->transaction_error_message = $error;

            return false;
        }
    }

    /**
     * Incluye las librerias externas para la
     * tokenización de las tarejtas.
     */
    public function enqueue_payment_scripts()
    {
        if (! is_checkout()) {
            return;
        }

        wp_enqueue_script('srpago_js', 'https://js.srpago.com/v1/srpago.min.js', '', '', true);
        wp_enqueue_script('srpago_encryption', 'https://js.srpago.com/v1/srpago.encryption.min.js', '', '', true);
        wp_enqueue_script('srpago', plugins_url('assets/js/srpago.jquery-payments.js', __FILE__), array('jquery'), '0.0.5', true);

        $params = array(
            'sandbox_mode'  => $this->sandbox,
            'public_key'    => $this->publicKey,
        );

        wp_localize_script('srpago', 'wc_srpago_params', $params);
    }
}
