<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Template Core Functions
 *
 * Template Core Functions specifically created for car listings
 *
 * @author 		Alex Gurghis
 * @category 	Core
 * @package 	autoroyal/Core Functions
 * @version     1.0.2
 */

/**
 * Get and include template files.
 *
 * @param mixed $template_name
 * @param array $args (default: array())
 * @param string $template_path (default: '')
 * @param string $default_path (default: '')
 * @return void
 */

/**
 * autoRoyal_Admin_Setup_Wizard class.
 */
class autoroyal_Admin_Setup_Wizard {

	/** @var string Currenct Step */
	private $step   = '';

	/** @var array Steps for the setup wizard */
	private $steps  = array();

	/**
	 * Hook in tabs.
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'admin_menus' ) );
		//add_action( 'admin_init', array( $this, 'setup_wizard' ) );
	}

	/**
	 * Add admin menus/screens.
	 */
	public function admin_menus() {
		add_dashboard_page( '', '', 'manage_options', 'autoroyal-setup', '' );
	}

	/**
	 * Show the setup wizard.
	 */
	public function setup_wizard() {
		if ( empty( $_GET['page'] ) || 'autoroyal-setup' !== $_GET['page'] ) {
			return;
		}
		$default_steps = array(
			'introduction' => array(
				'name'    => __( 'Introduction', 'autoroyal' ),
				'view'    => array( $this, 'autoroyal_setup_introduction' ),
				'handler' => '',
			),
			'type' => array(
				'name'    => __( 'Website Type', 'autoroyal' ),
				'view'    => array( $this, 'autoroyal_select_type' ),
				'handler' => array( $this, 'autoroyal_select_type_save' ),
			),
			'pages' => array(
				'name'    => __( 'Page Setup', 'autoroyal' ),
				'view'    => array( $this, 'autoroyal_setup_pages' ),
				'handler' => array( $this, 'autoroyal_setup_pages_save' ),
			),
			'locale' => array(
				'name'    => __( 'Website Locale', 'autoroyal' ),
				'view'    => array( $this, 'autoroyal_setup_locale' ),
				'handler' => array( $this, 'autoroyal_setup_locale_save' ),
			),
			'next_steps' => array(
				'name'    => __( 'All Done!', 'autoroyal' ),
				'view'    => array( $this, 'autoroyal_setup_ready' ),
				'handler' => '',
			),
		);

		$this->steps = apply_filters( 'autoroyal_setup_wizard_steps', $default_steps );
		$this->step = isset( $_GET['step'] ) ? sanitize_key( $_GET['step'] ) : current( array_keys( $this->steps ) );

		wp_register_script( 'jquery-blockui', autoroyal_PLUGIN_URL . '/assets/js/jquery.blockUI.min.js', array( 'jquery' ), '2.70', true );
		wp_register_script( 'select2', autoroyal_PLUGIN_URL . '/assets/js/select2.min.js', array( 'jquery' ), '4.0.3' );
		wp_register_script( 'wc-enhanced-select', autoroyal_PLUGIN_URL . '/assets/js/enhanced-select.min.js', array( 'jquery', 'select2' ) );
		wp_localize_script( 'wc-enhanced-select', 'autoroyal_enhanced_select_params', array(
			'i18n_no_matches'           => _x( 'No matches found', 'enhanced select', 'autoroyal' ),
			'i18n_ajax_error'           => _x( 'Loading failed', 'enhanced select', 'autoroyal' ),
			'i18n_input_too_short_1'    => _x( 'Please enter 1 or more characters', 'enhanced select', 'autoroyal' ),
			'i18n_input_too_short_n'    => _x( 'Please enter %qty% or more characters', 'enhanced select', 'autoroyal' ),
			'i18n_input_too_long_1'     => _x( 'Please delete 1 character', 'enhanced select', 'autoroyal' ),
			'i18n_input_too_long_n'     => _x( 'Please delete %qty% characters', 'enhanced select', 'autoroyal' ),
			'i18n_selection_too_long_1' => _x( 'You can only select 1 item', 'enhanced select', 'autoroyal' ),
			'i18n_selection_too_long_n' => _x( 'You can only select %qty% items', 'enhanced select', 'autoroyal' ),
			'i18n_load_more'            => _x( 'Loading more results&hellip;', 'enhanced select', 'autoroyal' ),
			'i18n_searching'            => _x( 'Searching&hellip;', 'enhanced select', 'autoroyal' ),
			'ajax_url'                  => admin_url( 'admin-ajax.php' ),
			'search_products_nonce'     => wp_create_nonce( 'search-products' ),
			'search_customers_nonce'    => wp_create_nonce( 'search-customers' ),
		) );
		wp_enqueue_style( 'autoroyal_admin_styles', autoroyal_PLUGIN_URL . '/assets/css/setup-wizard.css', array() );
		wp_enqueue_style( 'autoroyal-setup', autoroyal_PLUGIN_URL . '/assets/css/autoroyal-setup.css', array( 'dashicons', 'install' ) );
		wp_enqueue_style( 'fontawesome',         autoroyal_PLUGIN_URL . '/assets/css/font-awesome.min.css', array(), '4.5.0', 'all' );

		wp_register_script( 'autoroyal-setup', autoroyal_PLUGIN_URL . '/assets/js/autoroyal-setup.min.js', array( 'jquery', 'wc-enhanced-select', 'jquery-blockui' ) );
		wp_localize_script( 'autoroyal-setup', 'autoroyal_setup_params', array(
			'locale_info' => json_encode( include( autoroyal_PLUGIN_DIR . '/i18n/locale-info.php' ) ),
		) );

		if ( ! empty( $_POST['save_step'] ) && isset( $this->steps[ $this->step ]['handler'] ) ) {
			call_user_func( $this->steps[ $this->step ]['handler'], $this );
		}

		ob_start();
		$this->setup_wizard_header();
		$this->setup_wizard_steps();
		$this->setup_wizard_content();
		$this->setup_wizard_footer();
		exit;
	}

	/**
	 * Get the URL for the next step's screen.
	 * @param string step   slug (default: current step)
	 * @return string       URL for next step if a next step exists.
	 *                      Admin URL if it's the last step.
	 *                      Empty string on failure.
	 * @since 3.0.0
	 */
	public function get_next_step_link( $step = '' ) {
		if ( ! $step ) {
			$step = $this->step;
		}

		$keys = array_keys( $this->steps );
		if ( end( $keys ) === $step ) {
			return admin_url();
		}

		$step_index = array_search( $step, $keys );
		if ( false === $step_index ) {
			return '';
		}

		return add_query_arg( 'step', $keys[ $step_index + 1 ] );
	}

	/**
	 * Setup Wizard Header.
	 */
	public function setup_wizard_header() {
		?>
		<!DOCTYPE html>
		<html <?php language_attributes(); ?>>
		<head>
			<meta name="viewport" content="width=device-width" />
			<meta http-equiv="Content-Type" content="text/html; charset=utf-8" />
			<title><?php esc_html_e( 'autoroyal &rsaquo; Setup Wizard', 'autoroyal' ); ?></title>
			<?php wp_print_scripts( 'autoroyal-setup' ); ?>
			<?php do_action( 'admin_print_styles' ); ?>
			<?php do_action( 'admin_head' ); ?>
		</head>
		<body class="autoroyal-setup wp-core-ui">
			<h1 id="autoroyal-setup-wizard-logo">
				<a href="http://wpautoroyalplugin.com/">
					<img src="<?php echo autoroyal_PLUGIN_URL; ?>/assets/images/logo-setup-wizard-2.png" alt="autoroyal" />
				</a>
				<span><?php esc_html_e( 'Quick Setup Wizard', 'autoroyal' ); ?></span>
			</h1>
		<?php
	}

	/**
	 * Setup Wizard Footer.
	 */
	public function setup_wizard_footer() {
		// Nothing here
	}

	/**
	 * Output the steps.
	 */
	public function setup_wizard_steps() {
		$ouput_steps = $this->steps;
		array_shift( $ouput_steps );
		?>
		<ol class="autoroyal-setup-steps">
			<?php foreach ( $ouput_steps as $step_key => $step ) : ?>
				<li class="<?php
					if ( $step_key === $this->step ) {
						echo 'active';
					} elseif ( array_search( $this->step, array_keys( $this->steps ) ) > array_search( $step_key, array_keys( $this->steps ) ) ) {
						echo 'done';
					}
				?>"><?php echo esc_html( $step['name'] ); ?></li>
			<?php endforeach; ?>
		</ol>
		<?php
	}

	/**
	 * Output the content for the current step.
	 */
	public function setup_wizard_content() {
		echo '<div class="autoroyal-setup-content">';
		call_user_func( $this->steps[ $this->step ]['view'], $this );
		echo '</div>';
	}

	/**
	 * Introduction step.
	 */
	public function autoroyal_setup_introduction() {
		?>
		<h1><?php esc_html_e( 'Overview!', 'autoroyal' ); ?></h1>
		<p><?php _e( 'This quick setup wizard will help you <strong>configure the basic settings</strong> of the autoroyal WordPress Plugin. It shouldn’t take more than 5 minutes and it’s optional, though recomendable.', 'autoroyal' ); ?></p>
		<p><?php esc_html_e( 'If you’re in a hurry now, don’t worry, just skip it and return to the WordPress Admin Dashboard. You can get back to this when you have the time.', 'autoroyal' ); ?></p>
		<p><?php esc_html_e( 'Thank you for choosing autoroyal WordPress Plugin to power your vehicles shop or automotive marketplace.', 'autoroyal' ); ?></p>
		<p class="autoroyal-setup-actions step">
			<a href="<?php echo esc_url( $this->get_next_step_link() ); ?>" class="button-primary button button-large button-next button-padding-right"><?php esc_html_e( 'Let’s do this!', 'autoroyal' ); ?><i class="fa fa-angle-right" aria-hidden="true"></i></a>
			<a href="<?php echo esc_url( admin_url() ); ?>" class="button button-clean align-left"><?php esc_html_e( 'Not now, I’ll do this later', 'autoroyal' ); ?></a>
		</p>
		<?php
	}

	/**
	 * Select website type step.
	 */
	public function autoroyal_select_type() {
		?>
		<h1><?php esc_html_e( 'Website Type', 'autoroyal' ); ?></h1>
		<form method="post">
			<p><?php esc_html_e( 'Here you will select the type of your website, a vehicles shop (car dealer) or automotive marketplace.', 'autoroyal' ); ?></p>

			<div class="middle">

			  	<label>
				  	<input type="radio" value="dealer" name="website_type" checked/>
				  	<div class="front-end box">
				    	<span><?php esc_html_e( 'Vehicles Shop', 'autoroyal' ); ?></span>
				  	</div>
				</label>

				<label>
				  	<input type="radio" value="rentals" name="website_type" checked/>
				  	<div class="front-end box">
				    	<span><?php esc_html_e( 'Vehicles Rent', 'autoroyal' ); ?></span>
				  	</div>
				</label>

				<label>
			 	 	<input type="radio" value="marketplace" name="website_type"/>
				  	<div class="back-end box">
				    	<span><?php esc_html_e( 'Marketplace', 'autoroyal' ); ?></span>
				  	</div>
				</label>

			</div>


			<p class="autoroyal-setup-actions step">
				<input type="submit" class="button-primary button button-large button-next" value="<?php esc_attr_e( 'Continue', 'autoroyal' ); ?>" name="save_step" />
				<a href="<?php echo esc_url( $this->get_next_step_link() ); ?>" class="button button-large button-next"><?php esc_html_e( 'Skip this step', 'autoroyal' ); ?></a>
				<?php wp_nonce_field( 'autoroyal-setup' ); ?>
			</p>
		</form>
		<?php
	}

	/**
	 * Save website type.
	 */
	public function autoroyal_select_type_save() {
		check_admin_referer( 'autoroyal-setup' );

		$website_type = sanitize_text_field( $_POST['website_type'] );
		update_option( 'autoroyal_webiste_type', $website_type );

		wp_redirect( esc_url_raw( $this->get_next_step_link() ) );
		exit;
	}

	/**
	 * Page setup.
	 */
	public function autoroyal_setup_pages() {
		?>
		<h1><?php esc_html_e( 'Page setup', 'autoroyal' ); ?></h1>
		<form method="post">
			<p><?php printf( __( 'Your store needs a few essential <a href="%s" target="_blank">pages</a>. The following will be created automatically (if they do not already exist):', 'autoroyal' ), esc_url( admin_url( 'edit.php?post_type=page' ) ) ); ?></p>
			<table class="autoroyal-setup-pages" cellspacing="0">
				<thead>
					<tr>
						<th class="page-name"><?php esc_html_e( 'Page name', 'autoroyal' ); ?></th>
						<th class="page-description"><?php esc_html_e( 'Description', 'autoroyal' ); ?></th>
					</tr>
				</thead>
				<tbody>
					<tr>
						<td class="page-name"><?php echo _x( 'Add/Edit Vehicle', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end submit/edit vehicle page.', 'autoroyal' ); ?></td>
					</tr>
					<tr>
						<td class="page-name"><?php echo _x( 'Dealer Dashboard Page', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end dealer dashboard page.', 'autoroyal' ); ?></td>
					</tr>
					<tr>
						<td class="page-name"><?php echo _x( 'Dealer Inventory Page', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end dealer inventory page.', 'autoroyal' ); ?></td>
					</tr>
					<tr>
						<td class="page-name"><?php echo _x( 'Dealer Leads Page', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end dealer leads page.', 'autoroyal' ); ?></td>
					</tr>
					<tr>
						<td class="page-name"><?php echo _x( 'Add/Edit Lead Page', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end dealer submit/edit lead page.', 'autoroyal' ); ?></td>
					</tr>
					<tr>
						<td class="page-name"><?php echo _x( 'Dealer Deals Page', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end dealer deals page.', 'autoroyal' ); ?></td>
					</tr>
					<tr>
						<td class="page-name"><?php echo _x( 'Add/Edit Deal Page', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end dealer submit/edit deal page.', 'autoroyal' ); ?></td>
					</tr>
					<tr>
						<td class="page-name"><?php echo _x( 'Browse All Cars Page', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end all cars page.', 'autoroyal' ); ?></td>
					</tr>
					<tr>
						<td class="page-name"><?php echo _x( 'Reports', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end reports page.', 'autoroyal' ); ?></td>
					</tr>
					<tr>
						<td class="page-name"><?php echo _x( 'Upcoming Payments', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end upcoming payments page.', 'autoroyal' ); ?></td>
					</tr>
					<tr>
						<td class="page-name"><?php echo _x( 'Notifications', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end notifications page.', 'autoroyal' ); ?></td>
					</tr>
					<tr>
						<td class="page-name"><?php echo _x( 'Notifications Subscribers Page', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end notifications subscribers page.', 'autoroyal' ); ?></td>
					</tr>
					<tr>
						<td class="page-name"><?php echo _x( 'Account Settings Page', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end account settings page.', 'autoroyal' ); ?></td>
					</tr>
					<tr>
						<td class="page-name"><?php echo _x( 'Compare Vehicles Page', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end compare vehicles page.', 'autoroyal' ); ?></td>
					</tr>
					<tr>
						<td class="page-name"><?php echo _x( 'Search Vehicles Page', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end search vehicles page.', 'autoroyal' ); ?></td>
					</tr>

					<?php
						$website_type = get_option('autoroyal_webiste_type');
						if($website_type == "rentals") {
					?>
					<tr>
						<td class="page-name"><?php echo _x( 'Rent Reservations Page', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end rent reservations page.', 'autoroyal' ); ?></td>
					</tr>
					<tr>
						<td class="page-name"><?php echo _x( 'Add/Edit Reservation Page', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end add/edit rent reservation page.', 'autoroyal' ); ?></td>
					</tr>
					<tr>
						<td class="page-name"><?php echo _x( 'Search Vehicles For Rent Page', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end Search vehicles for rent page.', 'autoroyal' ); ?></td>
					</tr>
					<?php } ?>


					<?php
						$website_type = get_option('autoroyal_webiste_type');
						if($website_type == "dealer") {
					?>
					<tr>
						<td class="page-name"><?php echo _x( 'Filter Subscriptions', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end filter subscriptions page.', 'autoroyal' ); ?></td>
					</tr>
					<?php } ?>

					<?php
						$website_type = get_option('autoroyal_webiste_type');
						if($website_type == "marketplace") {
					?>
					<tr>
						<td class="page-name"><?php echo _x( 'User Profile Page', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end user profile page (for marketplace only).', 'autoroyal' ); ?></td>
					</tr>
					<tr>
						<td class="page-name"><?php echo _x( 'Account Funds', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end account funds page (for marketplace only).', 'autoroyal' ); ?></td>
					</tr>

					<tr>
						<td class="page-name"><?php echo _x( 'My Active Advertised Listings Page', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end my active advertised listings page.', 'autoroyal' ); ?></td>
					</tr>
					<tr>
						<td class="page-name"><?php echo _x( 'Settings', 'Page title', 'autoroyal' ); ?></td>
						<td><?php esc_html_e( 'This is front-end settings page.', 'autoroyal' ); ?></td>
					</tr>
					<?php } ?>
				</tbody>
			</table>

			<p><?php printf( __( 'Once created, these pages can be managed from your admin dashboard on the <a href="%1$s" target="_blank">Pages screen</a>. You can control which pages are shown on your website via <a href="%2$s" target="_blank">Appearance > Menus</a>.', 'autoroyal' ), esc_url( admin_url( 'edit.php?post_type=page' ) ), esc_url( admin_url( 'nav-menus.php' ) ) ); ?></p>

			<p class="autoroyal-setup-actions step">
				<input type="submit" class="button-primary button button-large button-next" value="<?php esc_attr_e( 'Continue', 'autoroyal' ); ?>" name="save_step" />
				<a href="<?php echo esc_url( $this->get_next_step_link() ); ?>" class="button button-large button-next"><?php esc_html_e( 'Skip this step', 'autoroyal' ); ?></a>
				<?php wp_nonce_field( 'autoroyal-setup' ); ?>
			</p>
		</form>
		<?php
	}

	/**
	 * Save Page Settings.
	 */
	public function autoroyal_setup_pages_save() {
		check_admin_referer( 'autoroyal-setup' );

		autoroyal_Install::create_pages();
		wp_redirect( esc_url_raw( $this->get_next_step_link() ) );
		exit;
	}

	/**
	 * Locale settings.
	 */
	public function autoroyal_setup_locale() {
		$user_location  = autoroyal_Geolocation::geolocate_ip();
		$country        = ! empty( $user_location['country'] ) ? $user_location['country'] : 'US';
		$state          = ! empty( $user_location['state'] ) ? $user_location['state'] : '*';
		$state          = 'US' === $country && '*' === $state ? 'AL' : $state;

		// Defaults
		$currency           = get_option( 'autoroyal_currency', 'USD' );
		$currency_pos       = get_option( 'autoroyal_currency_position', 'left' );
		$measurement_type   = get_option( 'autoroyal_measurement_type', 'metric' );
		$decimal_sep        = get_option( 'autoroyal_decimal_separator', '.' );
		$num_decimals       = get_option( 'autoroyal_price_num_decimals', '2' );
		$thousand_sep       = get_option( 'autoroyal_thousand_separator', ',' );
		?>
		<h1><?php esc_html_e( 'Store locale setup', 'autoroyal' ); ?></h1>
		<form method="post">
			<table class="form-table">
				<tr>
					<th scope="row"><label for="currency_pos"><?php esc_html_e( 'Systems of Measurement', 'autoroyal' ); ?></label></th>
					<td>
						<select id="measurement_type" name="measurement_type" class="wc-enhanced-select">
							<option value="metric" <?php selected( $measurement_type, 'metric' ); ?>><?php esc_html_e( 'Metric', 'autoroyal' ); ?></option>
							<option value="usa" <?php selected( $measurement_type, 'usa' ); ?>><?php esc_html_e( 'U.S.', 'autoroyal' ); ?></option>
						</select>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="currency_code"><?php esc_html_e( 'Currency', 'autoroyal' ); ?></label></th>
					<td>
						<select id="currency_code" name="currency_code" style="width:100%;" data-placeholder="<?php esc_attr_e( 'Choose a currency&hellip;', 'autoroyal' ); ?>" class="wc-enhanced-select">
							<option value=""><?php esc_html_e( 'Choose a currency&hellip;', 'autoroyal' ); ?></option>
							<?php
							foreach ( get_autoroyal_currencies() as $code => $name ) {
								echo '<option value="' . esc_attr( $code ) . '" ' . selected( $currency, $code, false ) . '>' . sprintf( esc_html__( '%1$s (%2$s)', 'autoroyal' ), $name, autoroyal_get_currency_symbol( $code ) ) . '</option>';
							}
							?>
						</select>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="currency_pos"><?php esc_html_e( 'Currency position', 'autoroyal' ); ?></label></th>
					<td>
						<select id="currency_pos" name="currency_pos" class="wc-enhanced-select">
							<option value="left" <?php selected( $currency_pos, 'left' ); ?>><?php esc_html_e( 'Left', 'autoroyal' ); ?></option>
							<option value="right" <?php selected( $currency_pos, 'right' ); ?>><?php esc_html_e( 'Right', 'autoroyal' ); ?></option>
							<option value="left_space" <?php selected( $currency_pos, 'left_space' ); ?>><?php esc_html_e( 'Left with space', 'autoroyal' ); ?></option>
							<option value="right_space" <?php selected( $currency_pos, 'right_space' ); ?>><?php esc_html_e( 'Right with space', 'autoroyal' ); ?></option>
						</select>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="thousand_sep"><?php esc_html_e( 'Thousand separator', 'autoroyal' ); ?></label></th>
					<td>
						<input type="text" id="thousand_sep" name="thousand_sep" size="2" value="<?php echo esc_attr( $thousand_sep ); ?>" />
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="decimal_sep"><?php esc_html_e( 'Decimal separator', 'autoroyal' ); ?></label></th>
					<td>
						<input type="text" id="decimal_sep" name="decimal_sep" size="2" value="<?php echo esc_attr( $decimal_sep ); ?>" />
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="num_decimals"><?php esc_html_e( 'Number of decimals', 'autoroyal' ); ?></label></th>
					<td>
						<input type="text" id="num_decimals" name="num_decimals" size="2" value="<?php echo esc_attr( $num_decimals ); ?>" />
					</td>
				</tr>
			</table>
			<p class="autoroyal-setup-actions step">
				<input type="submit" class="button-primary button button-large button-next" value="<?php esc_attr_e( 'Continue', 'autoroyal' ); ?>" name="save_step" />
				<a href="<?php echo esc_url( $this->get_next_step_link() ); ?>" class="button button-large button-next"><?php esc_html_e( 'Skip this step', 'autoroyal' ); ?></a>
				<?php wp_nonce_field( 'autoroyal-setup' ); ?>
			</p>
		</form>
		<?php
	}

	/**
	 * Save Locale Settings.
	 */
	public function autoroyal_setup_locale_save() {
		check_admin_referer( 'autoroyal-setup' );

		$currency_code     = sanitize_text_field( $_POST['currency_code'] );
		$currency_pos      = sanitize_text_field( $_POST['currency_pos'] );
		$measurement_type  = sanitize_text_field( $_POST['measurement_type'] );
		$decimal_sep       = sanitize_text_field( $_POST['decimal_sep'] );
		$num_decimals      = sanitize_text_field( $_POST['num_decimals'] );
		$thousand_sep      = sanitize_text_field( $_POST['thousand_sep'] );

		update_option( 'autoroyal_currency', $currency_code );
		update_option( 'autoroyal_currency_position', $currency_pos );
		update_option( 'autoroyal_measurement_type', $measurement_type );
		update_option( 'autoroyal_decimal_separator', $decimal_sep );
		update_option( 'autoroyal_price_num_decimals', $num_decimals );
		update_option( 'autoroyal_thousand_separator', $thousand_sep );

		wp_redirect( esc_url_raw( $this->get_next_step_link() ) );
		exit;
	}

	/**
	 * Final step.
	 */
	public function autoroyal_setup_ready() {
		
		?>

		<h1><?php esc_html_e( 'Well Done!', 'autoroyal' ); ?></h1>

		<p><?php esc_html_e( 'Thank you for choosing autoroyal WordPress Plugin to power your vehicles shop or automotive marketplace.', 'autoroyal' ); ?></p>
		<p class="autoroyal-setup-actions step">
			<a href="<?php echo esc_url( admin_url( 'post-new.php?post_type=vehicle' ) ); ?>" class="button-primary button button-large button-next button-padding-right"><?php esc_html_e( 'Submit your first vehicle!', 'autoroyal' ); ?><i class="fa fa-angle-right" aria-hidden="true"></i></a>
			<a href="<?php echo esc_url( admin_url() ); ?>" class="button button-large button-next"><?php esc_html_e( 'Return to the WordPress Dashboard', 'autoroyal' ); ?></a>
		</p>
		<?php
	}
}

new autoroyal_Admin_Setup_Wizard();
